package presentation;

import java.awt.*;
import javax.swing.*;
import abstraction.Service;
import abstraction.State;
import abstraction.Action;
import abstraction.StateAction;
import org.jgraph.JGraph;
import org.jgraph.graph.DefaultEdge;
import org.jgraph.graph.DefaultGraphCell;

import java.util.Iterator;
import java.util.HashMap;
import presentation.graph.GraphFactory;
import control.*;

/**
 * This class is the panel used to execute the orchestrator
 */

public class ExecuteOrchestratorPanel extends JPanel {

	private static final long serialVersionUID = -8112895237970803475L;

	private OrchestratorFrame owner = null;
	
	private JLabel orchestratorLabel = null;
	private JLabel targetPresentStateLabel = null;
	
	private JButton doActionButton = null;
	private JButton setStartStatesButton = null;
	
	private JScrollPane graphScrollPane = null;
	private JScrollPane targetActionsScrollPane = null;
	private JScrollPane availablePresentStatesScrollPane = null;
	private JScrollPane availableServicesScrollPane = null;
	
	private JTable targetActionsTable = null;
	private JTable availablePresentStatesTable = null;
	private JTable availableServicesTable = null;
	
	private CustomTableModel actionsModel;
	private CustomTableModel availablePresentStateModel;
	private CustomTableModel availableServicesModel;

	private JGraph graph = null;

	private static final String[] actionColumnName = {"Action", "Next State"};
	private static final String[] availablePresentStateColumnName = {"Service", "Present State"};
	private static final String[] availableServicesColumnName = {"Service"};
	
	private State presentStateTarget = null;
	private CState presentStateAvailableServices = null; 
	
	private Service targetService = null;
	private Community community;
	private Orchestrator orchestrator;
	private SimulatedBy sm = null;

	private HashMap<String, Integer> indexServices;
	private HashMap<State, DefaultGraphCell> stateTable;
	private HashMap<StateAction, DefaultEdge> actionTable;
	
	private DefaultEdge selectedEdge;


	/**
	 * This is the default constructor
	 */
	public ExecuteOrchestratorPanel(OrchestratorFrame frame) {
		super();
		initialize();
		owner = frame;
	}

	/**
	 * This method initializes this
	 * 
	 */
	private void initialize() {
		GridBagConstraints gridBagConstraints13 = new GridBagConstraints();
		gridBagConstraints13.gridx = 1;
		gridBagConstraints13.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints13.ipadx = 40;
		gridBagConstraints13.gridy = 0;
		GridBagConstraints gridBagConstraints12 = new GridBagConstraints();
		gridBagConstraints12.fill = GridBagConstraints.VERTICAL;
		gridBagConstraints12.gridy = 3;
		gridBagConstraints12.weightx = 0.0;
		gridBagConstraints12.weighty = 1.0;
		gridBagConstraints12.ipadx = 144;
		gridBagConstraints12.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints12.gridx = 0;
		GridBagConstraints gridBagConstraints11 = new GridBagConstraints();
		gridBagConstraints11.fill = GridBagConstraints.VERTICAL;
		gridBagConstraints11.gridy = 6;
		gridBagConstraints11.weightx = 0.0;
		gridBagConstraints11.weighty = 1.0;
		gridBagConstraints11.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints11.ipadx = 144;
		gridBagConstraints11.gridx = 0;
		GridBagConstraints gridBagConstraints4 = new GridBagConstraints();
		gridBagConstraints4.fill = GridBagConstraints.VERTICAL;
		gridBagConstraints4.gridy = 2;
		gridBagConstraints4.weightx = 0.0;
		gridBagConstraints4.weighty = 1.0;
		gridBagConstraints4.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints4.ipadx = 144;
		gridBagConstraints4.gridx = 0;
		GridBagConstraints gridBagConstraints3 = new GridBagConstraints();
		gridBagConstraints3.fill = GridBagConstraints.BOTH;
		gridBagConstraints3.gridy = 1;
		gridBagConstraints3.weightx = 1.0;
		gridBagConstraints3.weighty = 1.0;
		gridBagConstraints3.gridheight = 7;
		gridBagConstraints3.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints3.gridx = 1;
		GridBagConstraints gridBagConstraints2 = new GridBagConstraints();
		gridBagConstraints2.gridx = 0;
		gridBagConstraints2.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints2.ipadx = 45;
		gridBagConstraints2.gridy = 1;
		targetPresentStateLabel = new JLabel();
		targetPresentStateLabel.setText("Target Present State:");
		GridBagConstraints gridBagConstraints1 = new GridBagConstraints();
		gridBagConstraints1.gridx = 0;
		gridBagConstraints1.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints1.ipadx = 40;
		gridBagConstraints1.gridy = 5;
		GridBagConstraints gridBagConstraints = new GridBagConstraints();
		gridBagConstraints.gridx = 0;
		gridBagConstraints.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints.ipadx = 32;
		gridBagConstraints.gridy = 0;
		orchestratorLabel = new JLabel();
		orchestratorLabel.setText("Orchestrator Generator");
		this.setSize(300, 200);
		this.setLayout(new GridBagLayout());
		this.add(orchestratorLabel, gridBagConstraints);
		this.add(getDoActionButton(), gridBagConstraints1);
		this.add(targetPresentStateLabel, gridBagConstraints2);
		this.add(getGraphScrollPane(), gridBagConstraints3);
		this.add(getTargetActionsScrollPane(), gridBagConstraints4);
		this.add(getAvailablePresentStatesScrollPane(), gridBagConstraints11);
		this.add(getAvailableServicesScrollPane(), gridBagConstraints12);
		this.add(getSetStartStatesButton(), gridBagConstraints13);
	}

	/**
	 * This method initializes doActionButton	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getDoActionButton() {
		if (doActionButton == null) {
			doActionButton = new JButton();
			doActionButton.setText("Execute Action");
			doActionButton.setToolTipText("Execute the selected Action");
			doActionButton.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					/* 
					 * Execute the selected action
					 * change the current state of the target service
					 * change the current state of the selected available service
					 * 
					 * 
					 */
					
					int actionIndex = targetActionsTable.getSelectedRow();
					int serviceIndex = availableServicesTable.getSelectedRow();
					if(actionIndex < 0)
					{
						JOptionPane.showMessageDialog(null, "Select an action on the table", "Warning", JOptionPane.WARNING_MESSAGE);
					}
					else if(serviceIndex<0)
					{
						JOptionPane.showMessageDialog(null, "Select a service on the table", "Warning", JOptionPane.WARNING_MESSAGE);
					}
					else
					{
					
						GraphFactory.selectVertex(graph, stateTable.get(presentStateTarget), false);
						presentStateTarget= (State)actionsModel.getValueAt(actionIndex, 1);
						GraphFactory.selectVertex(graph, stateTable.get(presentStateTarget), true);
						GraphFactory.selectEdge(graph, selectedEdge, false);
						
						setLabelPresentState(presentStateTarget.getName());

						Action selectedAction = (Action)actionsModel.getValueAt(actionIndex, 0);

						Service selectedService = (Service)availableServicesModel.getValueAt(serviceIndex, 0);
						
						int index = indexServices.get(selectedService.getName());

						State presentAvailableState = presentStateAvailableServices.get(index);
						
						Iterator<State> it = selectedService.getNextStates(presentAvailableState, selectedAction);
						State nextState = it.next();
						presentStateAvailableServices = presentStateAvailableServices.buildNewState(index, nextState);
						availablePresentStateModel.setValueAt(nextState, index, 1);
						
						sm = new SimulatedBy(presentStateTarget, presentStateAvailableServices);

						actionsModel = new CustomTableModel(actionColumnName);
						targetActionsTable.setModel(actionsModel);
						
						availableServicesModel = new CustomTableModel(availableServicesColumnName);
						availableServicesTable.setModel(availableServicesModel);

						Iterator<Action> actions = targetService.getActions(presentStateTarget);
						while(actions.hasNext())
						{
							Action action = actions.next();
							Iterator<State> nextStates = targetService.getNextStates(presentStateTarget, action);
							while(nextStates.hasNext())
							{
								State next = nextStates.next();
								actionsModel.addRow(new Object[]{action, next});
							}
						}	
					}
				}
			});
		}
		return doActionButton;
	}

	/**
	 * This method initializes graphScrollPane	
	 * 	
	 * @return javax.swing.JScrollPane	
	 */
	private JScrollPane getGraphScrollPane() {
		if (graphScrollPane == null) {
			graphScrollPane = new JScrollPane();
		}
		return graphScrollPane;
	}

	/**
	 * This method initializes targetActionsScrollPane	
	 * 	
	 * @return javax.swing.JScrollPane	
	 */
	private JScrollPane getTargetActionsScrollPane() {
		if (targetActionsScrollPane == null) {
			targetActionsScrollPane = new JScrollPane();
			targetActionsScrollPane.setViewportView(getTargetActionsTable());
		}
		return targetActionsScrollPane;
	}

	/**
	 * This method initializes targetActionsTable	
	 * 	
	 * @return javax.swing.JTable	
	 */
	private JTable getTargetActionsTable() {
		if (targetActionsTable == null) {
			targetActionsTable = new JTable();
			targetActionsTable.setToolTipText("Choose an action");
			targetActionsTable.addMouseListener(new java.awt.event.MouseAdapter() {
				public void mouseClicked(java.awt.event.MouseEvent e) {
					int ind = targetActionsTable.getSelectedRow();
					Action action = (Action)actionsModel.getValueAt(ind, 0);
					OrchestratorKey ok = new OrchestratorKey(sm, action);
					Iterator<Service> s = orchestrator.getServicesForStateAction(ok);
					Service se = null;
					availableServicesModel = new CustomTableModel(availableServicesColumnName);
					while(s.hasNext())
					{
						
						se = s.next();
						availableServicesModel.addRow(new Object[]{se});
					}
					availableServicesTable.setModel(availableServicesModel);
					
					if(selectedEdge!=null)
						GraphFactory.selectEdge(graph, selectedEdge, false);
					selectedEdge = actionTable.get(new StateAction(presentStateTarget,action));
					GraphFactory.selectEdge(graph, selectedEdge, true);
				}
			});
		}
		return targetActionsTable;
	}
	
	/**
	 * set the label that show the current state of the target
	 * @param state the name of the current state
	 */
	private void setLabelPresentState(String state)
	{
		targetPresentStateLabel.setText("Target Present State: "+state);
	}
	
	/**
	 * Initialize the panel
	 * @param target the target service
	 * @param com the community that hold all available services
	 * @param orch the Orchestrator generated
	 */
	public void initExecuteOrchestratorPanel(Service target, Community com, Orchestrator orch)
	{
		targetService = target;
		community = com;
		orchestrator = orch;
		presentStateTarget = null;
		
		GraphFactory gf = new GraphFactory(targetService);
		graph = gf.getGraph();
		stateTable = gf.getVertexMap();
		actionTable = gf.getEdgeMap();
		
		graphScrollPane.setViewportView(graph);
		init(targetService.getInitialState(),community.getInitialStates());
		
	}
	
	private void init(State tis, CState ais)
	{
		actionsModel = new CustomTableModel(actionColumnName);
		targetActionsTable.setModel(actionsModel);
	
		availablePresentStateModel = new CustomTableModel(availablePresentStateColumnName);
		availablePresentStatesTable.setModel(availablePresentStateModel);

		availableServicesModel = new CustomTableModel(availableServicesColumnName);
		availableServicesTable.setModel(availableServicesModel);
		
		presentStateAvailableServices = ais;
		
		if(presentStateTarget != null)
			GraphFactory.selectVertex(graph, stateTable.get(presentStateTarget), false);	
		presentStateTarget = tis;
		GraphFactory.selectVertex(graph, stateTable.get(presentStateTarget), true);
		setLabelPresentState(presentStateTarget.getName());
		
		if(selectedEdge!=null)
			GraphFactory.selectEdge(graph, selectedEdge, false);
		
		sm = new SimulatedBy(presentStateTarget, presentStateAvailableServices);

		//init table for actions of target
		Iterator<Action> actions = targetService.getActions(presentStateTarget);
		while(actions.hasNext())
		{
			Action action = actions.next();
			Iterator<State> nextStates = targetService.getNextStates(presentStateTarget, action);
			while(nextStates.hasNext())
			{
				State next = nextStates.next();
				actionsModel.addRow(new Object[]{action, next});
			}
		}
		
		//init table available services
		indexServices = new HashMap<String, Integer>(community.size());
		for(int i=0; i< community.size();i++)
		{
			Service s = community.getService(i);
			State is = presentStateAvailableServices.get(i);
			availablePresentStateModel.addRow(new Object[]{s,is});
			indexServices.put(s.getName(), i);
		}
	}


	/**
	 * This method initializes availablePresentStatesScrollPane	
	 * 	
	 * @return javax.swing.JScrollPane	
	 */
	private JScrollPane getAvailablePresentStatesScrollPane() {
		if (availablePresentStatesScrollPane == null) {
			availablePresentStatesScrollPane = new JScrollPane();
			availablePresentStatesScrollPane.setViewportView(getAvailablePresentStatesTable());
		}
		return availablePresentStatesScrollPane;
	}

	/**
	 * This method initializes availablePresentStatesTable	
	 * 	
	 * @return javax.swing.JTable	
	 */
	private JTable getAvailablePresentStatesTable() {
		if (availablePresentStatesTable == null) {
			availablePresentStatesTable = new JTable();
			availablePresentStatesTable.setToolTipText("Current states for all available services");
		}
		return availablePresentStatesTable;
	}

	/**
	 * This method initializes availableServicesScrollPane	
	 * 	
	 * @return javax.swing.JScrollPane	
	 */
	private JScrollPane getAvailableServicesScrollPane() {
		if (availableServicesScrollPane == null) {
			availableServicesScrollPane = new JScrollPane();
			availableServicesScrollPane.setViewportView(getAvailableServicesTable());
		}
		return availableServicesScrollPane;
	}

	/**
	 * This method initializes availableServicesTable	
	 * 	
	 * @return javax.swing.JTable	
	 */
	private JTable getAvailableServicesTable() {
		if (availableServicesTable == null) {
			availableServicesTable = new JTable();
			availableServicesTable.setToolTipText("Choose an available service");
		}
		return availableServicesTable;
	}

	/**
	 * This method initializes setInitialStatesButton	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getSetStartStatesButton() {
		if (setStartStatesButton == null) {
			setStartStatesButton = new JButton();
			setStartStatesButton.setText("Choose starting states");
			setStartStatesButton.setToolTipText("Choose the starting stetes for target and available services");
			setStartStatesButton.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					StartStatesDialog in = new StartStatesDialog(owner, targetService, community);
					State its = in.getTargetStartState();
					CState ias = in.getAvailableStartStates();
					if(its!=null && ias!=null)
					{
						init(its,ias);
					}
				}
			});
		}
		return setStartStatesButton;
	}
}